import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:rounded_loading_button_plus/rounded_loading_button.dart';
import 'package:wordpress_app/blocs/config_bloc.dart';
import 'package:wordpress_app/blocs/user_bloc.dart';
import 'package:wordpress_app/models/user.dart';
import 'package:wordpress_app/pages/done.dart';
import 'package:wordpress_app/pages/create_account.dart';
import 'package:wordpress_app/pages/forgot_password.dart';
import 'package:wordpress_app/services/app_service.dart';
import 'package:wordpress_app/utils/snacbar.dart';
import 'package:wordpress_app/widgets/social_logins.dart';
import '../models/icon_data.dart';
import '../services/auth_service.dart';
import '../utils/next_screen.dart';
import 'package:easy_localization/easy_localization.dart';

class LoginPage extends ConsumerStatefulWidget {
  const LoginPage({super.key, this.popUpScreen});

  final bool? popUpScreen;

  @override
  ConsumerState<LoginPage> createState() => _LoginPageState();
}

class _LoginPageState extends ConsumerState<LoginPage> {
  var formKey = GlobalKey<FormState>();
  var userNameCtrl = TextEditingController();
  var passwordCtrl = TextEditingController();
  final _btnController = RoundedLoadingButtonController();

  bool offsecureText = true;
  Icon lockIcon = LockIcon.lock;

  Future _handleLoginWithUsernamePassword() async {
    final UserBloc ub = ref.read(userBlocProvider);
    if (userNameCtrl.text.isEmpty) {
      _btnController.reset();
      openSnacbar(context, 'Username is required');
    } else if (passwordCtrl.text.isEmpty) {
      _btnController.reset();
      openSnacbar(context, 'Password is required');
    } else {
      FocusScope.of(context).unfocus();
      AppService().checkInternet().then((hasInternet) async {
        if (!hasInternet) {
          _btnController.reset();
          if (!mounted) return;
          openSnacbar(context, 'no-internet'.tr());
        } else {
          await AuthService.loginWithEmail(userNameCtrl.text, passwordCtrl.text).then((UserModel? userModel) async {
            if (userModel != null) {
              _btnController.reset();
              ub.guestUserSignout().then((value) => ub.saveUserData(userModel, 'email')).then((value) => ub.setSignIn()).then((value) {
                _btnController.success();
                afterSignIn();
              });
            } else {
              _btnController.reset();
              if (!mounted) return;
              openSnacbar(context, 'Username or password is invalid');
            }
          });
        }
      });
    }
  }

  void _onlockPressed() {
    if (offsecureText == true) {
      setState(() {
        offsecureText = false;
        lockIcon = LockIcon.open;
      });
    } else {
      setState(() {
        offsecureText = true;
        lockIcon = LockIcon.lock;
      });
    }
  }

  void afterSignIn() async {
    if (widget.popUpScreen == null || widget.popUpScreen == false) {
      nextScreenReplaceAnimation(context, const DonePage());
    } else {
      Navigator.pop(context);
    }
  }

  @override
  Widget build(BuildContext context) {
    final configs = ref.read(configBlocProvider).configs!;
    return Scaffold(
      resizeToAvoidBottomInset: true,
      backgroundColor: Theme.of(context).canvasColor,
      appBar: AppBar(
        elevation: 0,
        backgroundColor: Colors.transparent,
        leading: IconButton(
          icon: Icon(
            Icons.close,
            color: Theme.of(context).colorScheme.onSurface,
          ),
          onPressed: () => Navigator.pop(context),
        ),
      ),
      body: SafeArea(
        child: SingleChildScrollView(
          physics: const BouncingScrollPhysics(),
          padding: const EdgeInsets.symmetric(horizontal: 24, vertical: 20),
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.stretch,
            children: [
              Text(
                'login',
                style: Theme.of(context).textTheme.headlineLarge?.copyWith(
                      fontWeight: FontWeight.w800,
                    ),
              ).tr(),
              const SizedBox(height: 8),
              Text(
                'login-to-access-features',
                style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                      color: Theme.of(context).colorScheme.secondary,
                      fontSize: 15,
                    ),
              ).tr(),

              // Social Login Section
              Visibility(
                visible: configs.socialLoginsEnabled,
                child: Column(
                  children: [
                    SocialLogins(
                      afterSignIn: afterSignIn,
                    ),
                    const SizedBox(height: 30),
                    Row(
                      children: [
                        Expanded(
                          child: Divider(
                            thickness: 1,
                            height: 30,
                            color: Theme.of(context).colorScheme.outline.withValues(alpha: 0.3),
                          ),
                        ),
                        Padding(
                          padding: const EdgeInsets.symmetric(horizontal: 16),
                          child: Text(
                            'or',
                            style: TextStyle(
                              color: Theme.of(context).colorScheme.secondary,
                              fontSize: 14,
                              fontWeight: FontWeight.w500,
                            ),
                          ),
                        ),
                        Expanded(
                          child: Divider(
                            thickness: 1,
                            color: Theme.of(context).colorScheme.outline.withValues(alpha: 0.3),
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 32),
                  ],
                ),
              ),

              // Form Fields
              Column(
                crossAxisAlignment: CrossAxisAlignment.stretch,
                children: [
                  // Username/Email Field
                  Text(
                    'username/email',
                    style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                          fontWeight: FontWeight.w600,
                          fontSize: 14,
                        ),
                  ).tr(),
                  const SizedBox(height: 8),
                  Container(
                    decoration: BoxDecoration(
                      borderRadius: BorderRadius.circular(12),
                      border: Border.all(
                        color: Theme.of(context).colorScheme.outline.withValues(alpha: 0.2),
                        width: 1,
                      ),
                    ),
                    child: TextFormField(
                      controller: userNameCtrl,
                      keyboardType: TextInputType.emailAddress,
                      style: Theme.of(context).textTheme.bodyLarge,
                      decoration: InputDecoration(
                        hintText: 'enter-username-or-email'.tr(),
                        hintStyle: TextStyle(
                          color: Theme.of(context).colorScheme.secondary.withValues(alpha: 0.6),
                          fontSize: 15,
                          fontWeight: FontWeight.w400,
                        ),
                        border: InputBorder.none,
                        enabledBorder: InputBorder.none,
                        focusedBorder: InputBorder.none,
                        contentPadding: const EdgeInsets.symmetric(horizontal: 16, vertical: 16),
                        prefixIcon: Icon(
                          Icons.person_outline,
                          size: 22,
                          color: Theme.of(context).colorScheme.secondary.withValues(alpha: 0.7),
                        ),
                      ),
                    ),
                  ),
                  const SizedBox(height: 24),

                  // Password Field
                  Text(
                    'password',
                    style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                          fontWeight: FontWeight.w600,
                          fontSize: 14,
                        ),
                  ).tr(),
                  const SizedBox(height: 8),
                  Container(
                    decoration: BoxDecoration(
                      borderRadius: BorderRadius.circular(12),
                      border: Border.all(
                        color: Theme.of(context).colorScheme.outline.withValues(alpha: 0.2),
                        width: 1,
                      ),
                    ),
                    child: TextFormField(
                      controller: passwordCtrl,
                      obscureText: offsecureText,
                      keyboardType: TextInputType.text,
                      style: Theme.of(context).textTheme.bodyLarge,
                      decoration: InputDecoration(
                        hintText: 'enter-password'.tr(),
                        hintStyle: TextStyle(
                          color: Theme.of(context).colorScheme.secondary.withValues(alpha: 0.6),
                          fontSize: 15,
                          fontWeight: FontWeight.w400,
                        ),
                        border: InputBorder.none,
                        enabledBorder: InputBorder.none,
                        focusedBorder: InputBorder.none,
                        contentPadding: const EdgeInsets.symmetric(horizontal: 16, vertical: 16),
                        prefixIcon: Icon(
                          Icons.lock_outline,
                          size: 22,
                          color: Theme.of(context).colorScheme.secondary.withValues(alpha: 0.7),
                        ),
                        suffixIcon: IconButton(
                          icon: lockIcon,
                          onPressed: () => _onlockPressed(),
                          color: Theme.of(context).colorScheme.secondary.withValues(alpha: 0.7),
                        ),
                      ),
                    ),
                  ),
                  const SizedBox(height: 12),

                  // Forgot Password
                  Align(
                    alignment: Alignment.centerRight,
                    child: TextButton(
                      onPressed: () => nextScreeniOS(context, const ForgotPasswordPage()),
                      style: TextButton.styleFrom(
                        padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 4),
                        minimumSize: Size.zero,
                        tapTargetSize: MaterialTapTargetSize.shrinkWrap,
                      ),
                      child: Text(
                        'forgot-password',
                        style: TextStyle(
                          fontWeight: FontWeight.w600,
                          fontSize: 14,
                          color: Theme.of(context).primaryColor,
                        ),
                      ).tr(),
                    ),
                  ),
                  const SizedBox(height: 24),

                  // Login Button
                  RoundedLoadingButton(
                    animateOnTap: true,
                    controller: _btnController,
                    onPressed: () => _handleLoginWithUsernamePassword(),
                    width: MediaQuery.of(context).size.width * 1.0,
                    color: Theme.of(context).primaryColor,
                    elevation: 0,
                    borderRadius: 12,
                    height: 52,
                    child: Text(
                      'login',
                      style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.w700,
                            color: Colors.white,
                            fontSize: 16,
                          ),
                    ).tr(),
                  ),
                ],
              ),

              const SizedBox(height: 10),

              // Create Account Section
              Container(
                padding: const EdgeInsets.symmetric(vertical: 20),
                child: Row(
                  mainAxisAlignment: MainAxisAlignment.center,
                  children: [
                    Text(
                      "no-account",
                      style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                            color: Theme.of(context).colorScheme.secondary,
                            fontSize: 15,
                          ),
                    ).tr(),
                    const SizedBox(width: 4),
                    TextButton(
                      onPressed: () => nextScreenReplaceiOS(
                        context,
                        CreateAccountPage(
                          popUpScreen: widget.popUpScreen,
                        ),
                      ),
                      style: TextButton.styleFrom(
                        padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 4),
                        minimumSize: Size.zero,
                        tapTargetSize: MaterialTapTargetSize.shrinkWrap,
                      ),
                      child: Text(
                        'create-account',
                        style: Theme.of(context).textTheme.titleMedium?.copyWith(
                              fontWeight: FontWeight.w700,
                              fontSize: 15,
                              color: Theme.of(context).primaryColor,
                            ),
                      ).tr(),
                    ),
                  ],
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }
}
